<?php

namespace App\Models\Accounts\Income;

use App\Traits\Media;
use App\Traits\DateTime;
use App\Traits\Currencies;
use Illuminate\Database\Eloquent\Model;
use App\Models\Accounts\Setting\Category;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Revenue extends Model
{
    use HasFactory, Currencies, DateTime, Media;

    protected $table = 'ac_revenues';

    protected $dates = ['deleted_at', 'paid_at'];

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = ['company_id', 'account_id', 'paid_at', 'amount', 'currency_code', 'currency_rate', 'customer_id', 'description', 'category_id', 'payment_method', 'reference'];

    /**
     * Sortable columns.
     *
     * @var array
     */
    public $sortable = ['paid_at', 'amount','category_id', 'account', 'payment_method'];

    /**
     * Searchable rules.
     *
     * @var array
     */
    protected $searchableColumns = [
        'invoice_number' => 10,
        'order_number'   => 10,
        'customer_name'  => 10,
        'customer_email' => 5,
        'notes'          => 2,
    ];

    public function user()
    {
        return $this->belongsTo('App\Models\Accounts\Auth\User', 'customer_id', 'id');
    }

    public function account()
    {
        return $this->belongsTo('App\Models\Accounts\Banking\Account');
    }

    public function currency()
    {
        return $this->belongsTo('App\Models\Accounts\Setting\Currency', 'currency_code', 'code');
    }

    public function category()
    {
        return $this->belongsTo('App\Models\Accounts\Setting\Category');
    }

    public function customer()
    {
        return $this->belongsTo('App\Models\Accounts\Income\Customer');
    }

    public function transfers()
    {
        return $this->hasMany('App\Models\Accounts\Banking\Transfer');
    }

    /**
     * Get only transfers.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeIsTransfer($query)
    {
        return $query->where('category_id', '=', Category::transfer());
    }

    /**
     * Skip transfers.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeIsNotTransfer($query)
    {
        return $query->where('category_id', '<>', Category::transfer());
    }

    /**
     * Convert amount to double.
     *
     * @param  string  $value
     * @return void
     */
    public function setAmountAttribute($value)
    {
        $this->attributes['amount'] = (double) $value;
    }

    /**
     * Convert currency rate to double.
     *
     * @param  string  $value
     * @return void
     */
    public function setCurrencyRateAttribute($value)
    {
        $this->attributes['currency_rate'] = (double) $value;
    }

    public function scopeLatest($query)
    {
        return $query->orderBy('paid_at', 'desc');
    }

    /**
     * Get the current balance.
     *
     * @return string
     */
    public function getAttachmentAttribute($value)
    {
        if (!empty($value) && !$this->hasMedia('attachment')) {
            return $value;
        } elseif (!$this->hasMedia('attachment')) {
            return false;
        }

        return $this->getMedia('attachment')->last();
    }
}
