<?php

namespace App\Models\Accounts\Expense;

use App\Traits\Media;
use App\Traits\DateTime;
use App\Traits\Currencies;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class ChequePayment extends Model
{
    use HasFactory, Currencies, DateTime, Media;

    protected $table = 'ac_cheques_payments';

    protected $dates = ['deleted_at', 'paid_at'];

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = ['company_id', 'cheque_id', 'account_id', 'paid_at', 'amount', 'currency_code', 'currency_rate', 'description', 'payment_method', 'reference'];

    public function account()
    {
        return $this->belongsTo('App\Models\Accounts\Banking\Account');
    }

    public function currency()
    {
        return $this->belongsTo('App\Models\Accounts\Setting\Currency', 'currency_code', 'code');
    }

    public function cheque()
    {
        return $this->belongsTo('App\Models\Accounts\Expense\Cheque');
    }

    public function item()
    {
        return $this->belongsTo('App\Models\Accounts\Item\Item');
    }

    public function tax()
    {
        return $this->belongsTo('App\Models\Accounts\Setting\Tax');
    }

    public function scopeLatest($query)
    {
        return $query->orderBy('paid_at', 'desc');
    }

    /**
     * Convert amount to double.
     *
     * @param  string  $value
     * @return void
     */
    public function setAmountAttribute($value)
    {
        $this->attributes['amount'] = (double) $value;
    }

    /**
     * Convert currency rate to double.
     *
     * @param  string  $value
     * @return void
     */
    public function setCurrencyRateAttribute($value)
    {
        $this->attributes['currency_rate'] = (double) $value;
    }

    /**
     * Scope paid invoice.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePaid($query)
    {
        return $query->sum('amount');
    }

    /**
     * Get the current balance.
     *
     * @return string
     */
    public function getAttachmentAttribute($value)
    {
        if (!empty($value) && !$this->hasMedia('attachment')) {
            return $value;
        } elseif (!$this->hasMedia('attachment')) {
            return false;
        }

        return $this->getMedia('attachment')->last();
    } 
}
