<?php

namespace App\Models\Accounts\Expense;

use App\Traits\Media;
use App\Traits\DateTime;
use App\Traits\Currencies;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Cheque extends Model
{
    use HasFactory, Currencies, DateTime, Media;
  	protected $table = 'ac_cheques';

    protected $dates = ['deleted_at', 'paid_at', 'due_at'];

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = ['company_id','account_id','cheque_number', 'currency_code', 'currency_rate','cheque_status_code', 'paid_at', 'due_at', 'amount','person_id', 'person_name', 'person_email','person_phone', 'person_address', 'description','person_type'];

    /**
     * Searchable rules.
     *
     * @var array
     */
    protected $searchableColumns = [
        'cheque_number'    => 10,
        'person_name'    => 10,
        'person_email'   => 5,
        'person_phone'   => 2,
        'person_address' => 1,
        'description'    => 2,
    ];

    /**
     * Clonable relationships.
     *
     * @var array
     */
    protected $cloneable_relations = ['account_details', 'totals'];

    public function vendor()
    {
        return $this->belongsTo('App\Models\Accounts\Expense\Vendor');
    }

    public function currency()
    {
        return $this->belongsTo('App\Models\Accounts\Setting\Currency', 'currency_code', 'code');
    }

    public function status()
    {
        return $this->belongsTo('App\Models\Accounts\Expense\BillStatus', 'cheque_status_code', 'code');
    }

    public function items()
    {
        return $this->hasMany('App\Models\Accounts\Expense\ChequeItem');
    }

    public function account_details()
    {
        return $this->hasMany('App\Models\Accounts\Expense\ChequeAccount');
    }


    public function totals()
    {
        return $this->hasMany('App\Models\Accounts\Expense\ChequeTotal');
    }

    public function payments()
    {
        return $this->hasMany('App\Models\Accounts\Expense\ChequePayment');
    }

    public function histories()
    {
        return $this->hasMany('App\Models\Accounts\Expense\ChequeHistory');
    }

    public function scopeDue($query, $date)
    {
        return $query->where('due_at', '=', $date);
    }

    public function scopeLatest($query)
    {
        return $query->orderBy('paid_at', 'desc');
    }

    public function scopeAccrued($query)
    {
        return $query->where('cheque_status_code', '<>', 'draft');
    }

    public function scopePaid($query)
    {
        return $query->where('cheque_status_code', '=', 'paid');
    }

    public function scopeNotPaid($query)
    {
        return $query->where('cheque_status_code', '<>', 'paid');
    }

    public function onCloning($src, $child = null)
    {
        $this->cheque_status_code = 'draft';
        $this->cheque_number = $this->getNextChequeNumber();
    }

    /**
     * Convert amount to double.
     *
     * @param  string  $value
     * @return void
     */
    public function setAmountAttribute($value)
    {
        $this->attributes['amount'] = (double) $value;
    }

    /**
     * Convert currency rate to double.
     *
     * @param  string  $value
     * @return void
     */
    public function setCurrencyRateAttribute($value)
    {
        $this->attributes['currency_rate'] = (double) $value;
    }

    /**
     * Get the current balance.
     *
     * @return string
     */
    public function getAttachmentAttribute($value)
    {
        if (!empty($value) && !$this->hasMedia('attachment')) {
            return $value;
        } elseif (!$this->hasMedia('attachment')) {
            return false;
        }

        return $this->getMedia('attachment')->last();
    }
}
