<?php

namespace App\Models\Accounts\Expense;

use App\Traits\Media;
use App\Traits\DateTime;
use App\Traits\Currencies;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Bill extends Model
{
    use HasFactory, Currencies, DateTime, Media;

    protected $table = 'bills';

    protected $dates = ['deleted_at', 'billed_at', 'due_at'];

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = ['company_id', 'bill_number', 'order_number', 'bill_status_code', 'billed_at', 'due_at', 'amount', 'currency_code', 'currency_rate', 'vendor_id', 'vendor_name', 'vendor_email', 'vendor_tax_number', 'vendor_phone', 'vendor_address', 'notes','net_term'];

    /**
     * Sortable columns.
     *
     * @var array
     */
    public $sortable = ['bill_number', 'vendor_name', 'amount', 'status.name', 'billed_at', 'due_at', 'bill_status_code'];

    /**
     * Searchable rules.
     *
     * @var array
     */
    protected $searchableColumns = [
        'bill_number'    => 10,
        'order_number'   => 10,
        'vendor_name'    => 10,
        'vendor_email'   => 5,
        'vendor_phone'   => 2,
        'vendor_address' => 1,
        'notes'          => 2,
    ];

    /**
     * Clonable relationships.
     *
     * @var array
     */
    protected $cloneable_relations = ['items', 'totals','account_details'];

    public function vendor()
    {
        return $this->belongsTo('App\Models\Accounts\Expense\Vendor');
    }

    public function currency()
    {
        return $this->belongsTo('App\Models\Accounts\Setting\Currency', 'currency_code', 'code');
    }

     public function histories()
    {
        return $this->hasMany('App\Models\Accounts\Expense\BillHistory');
    }

    public function status()
    {
        return $this->belongsTo('App\Models\Accounts\Expense\BillStatus', 'bill_status_code', 'code');
    }

    public function items()
    {
        return $this->hasMany('App\Models\Accounts\Expense\BillItem');
    }
    public function account_details()
    {
        return $this->hasMany('App\Models\Accounts\Expense\BillAccount');
    }

    public function totals()
    {
        return $this->hasMany('App\Models\Accounts\Expense\BillTotal');
    }

    public function payments()
    {
        return $this->hasMany('App\Models\Accounts\Expense\BillPayment');
    }

        public function payments_total()
    {
        return $this->payments()->selectRaw('bill_payments.bill_id , sum(amount) as paid_amount')->groupBy('bill_payments.bill_id');
    }

    /* public function histories()
    {
        return $this->morphMany('App\Models\Accounts\Expense\BillHistory');
    }*/

    public function scopeDue($query, $date)
    {
        
        return $query->where('due_at', '=', $date);
    }

    public function scopeLatest($query)
    {
        return $query->orderBy('paid_at', 'desc');
    }

    public function scopeAccrued($query)
    {
        return $query->where('bill_status_code', '<>', 'draft');
    }

    public function scopePaid($query)
    {
        return $query->where('bill_status_code', '=', 'paid');
    }

    public function scopeNotPaid($query)
    {
        return $query->where('bill_status_code', '<>', 'paid');
    }

    public function onCloning($src, $child = null)
    {
        $this->bill_status_code = 'draft';
    }

    /**
     * Convert amount to double.
     *
     * @param  string  $value
     * @return void
     */
    public function setAmountAttribute($value)
    {
        $this->attributes['amount'] = (double) $value;
    }

    /**
     * Convert currency rate to double.
     *
     * @param  string  $value
     * @return void
     */
    public function setCurrencyRateAttribute($value)
    {
        $this->attributes['currency_rate'] = (double) $value;
    }

    /**
     * Get the current balance.
     *
     * @return string
     */
    public function getAttachmentAttribute($value)
    {
        if (!empty($value) && !$this->hasMedia('attachment')) {
            return $value;
        } elseif (!$this->hasMedia('attachment')) {
            return false;
        }

        return $this->getMedia('attachment')->last();
    }
}
