<?php

namespace App\Models\Accounts\Banking;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Account extends Model
{
    use HasFactory;

    protected $table = 'ac_accounts';

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['balance','name_number'];

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = ['company_id', 'name', 'number', 'currency_code', 'opening_balance', 'bank_name', 'bank_phone', 'bank_address', 'enabled','balance_as_of'];

    /**
     * Sortable columns.
     *
     * @var array
     */
    public $sortable = ['name', 'number', 'opening_balance', 'enabled'];

    /**
     * Searchable rules.
     *
     * @var array
     */
    protected $searchableColumns = [
        'name'         => 10,
        'number'       => 10,
        'bank_name'    => 10,
        'bank_phone'   => 5,
        'bank_address' => 2,
    ];

    public function currency()
    {
        return $this->belongsTo('App\Models\Accounts\Setting\Currency', 'currency_code', 'code');
    }

    public function invoice_payments()
    {
        return $this->hasMany('App\Models\Accounts\Income\InvoicePayment', 'account_id');
    }

    public function revenues()
    {
        
        return $this->hasMany('App\Models\Accounts\Income\Revenue');
    }

    public function deposite(){

        return $this->hasMany('App\Models\Accounts\Banking\Deposit');
    }

    public function bill_payments()
    {
        return $this->hasMany('App\Models\Accounts\Expense\BillPayment');
    }

    public function check_payments()
    {
        return $this->hasMany('App\Models\Accounts\Expense\Cheque','account_id','id');
    }

    public function payments()
    {
        return $this->hasMany('App\Models\Accounts\Expense\Payment');
    }

    public function chartofaccount(){

        return $this->hasOne('App\Models\Accounts\Accounting\Chartofaccounts','bank_id','id');
    }

    /**
     * Convert opening balance to double.
     *
     * @param  string  $value
     * @return void
     */
    public function setOpeningBalanceAttribute($value)
    {
        $this->attributes['opening_balance'] = (double) $value;
    }

    public function getNameNumberAttribute(){
       
       return  $this->name . '-' . substr($this->number, -4);
    }

    /**
     * Get the current balance.
     *
     * @return string
     */
    public function getBalanceAttribute()
    {

        // Opening Balance
        $total = $this->opening_balance;

        // Sum invoices
        foreach ($this->invoice_payments as $item) {
            $total += $item->amount;
        }

        // Sum revenues
        // foreach ($this->revenues as $item) {
        //     $total += $item->amount;
        // }
        // Sum Depostite
        foreach ($this->deposite as $item) {
            $total += $item->price;
        }
        // Bank Expenses Depostite
        if($this->chartofaccount){
            foreach ($this->chartofaccount->deposite_payments as $item) {
                $total -= $item->price;
            }
        }
        // Add Bank Checks
        if($this->chartofaccount){
            foreach ($this->chartofaccount->check_account as $item) {
                $total += $item->price;
            }
        }   

        // Subtract bills
        foreach ($this->bill_payments as $item) {
            $total -= $item->amount;
        }

        // Subtract payments
        foreach ($this->payments as $item) {
            $total -= $item->amount;
        }

        // subtract checks
        foreach($this->check_payments as $item){
            $total -= $item->amount;
        }
        return $total;
    }

    public function chatHistory()
    {
        return $this->hasMany('App\Models\Accounts\Accounting\Charthistory', 'transaction_id')->whereTransaction_type('Deposite');
    }

    public function chatofHistory()
    {
        return $this->hasMany('App\Models\Accounts\Accounting\Charthistory', 'chartofaccount_id');
    }

    public function reconciles()
    {
        return $this->hasMany('App\Models\Accounts\Banking\Reconcile', 'account_id');
    }
}
