<?php

namespace App\Http\Controllers\Accounts\Expenses;

use App\Traits\Uploads;
use App\Utilities\Modules;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Accounts\Income\Terms;
use App\Models\Accounts\Expense\Vendor;
use Illuminate\Support\Facades\Session;
use App\Models\Accounts\Banking\Account;
use App\Models\Accounts\Expense\Payment;
use App\Models\Accounts\Setting\Category;
use App\Models\Accounts\Setting\Currency;

class Payments extends Controller
{
    use Uploads;

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function index()
    {
        $payments = Payment::with(['vendor', 'account', 'category'])->get()->collect(['paid_at'=> 'desc']);

        $vendors = collect(Vendor::whereEnabled(1)->pluck('name', 'id'))
            ->prepend(trans('general.all_type', ['type' => trans_choice('general.vendors', 2)]), '');

        $categories = collect(Category::whereEnabled(1)->type('expense')->pluck('name', 'id'))
            ->prepend(trans('general.all_type', ['type' => trans_choice('general.categories', 2)]), '');

        $accounts = collect(Account::whereEnabled(1)->get()->pluck('name_number', 'id'))
            ->prepend(trans('general.all_type', ['type' => trans_choice('general.accounts', 2)]), '');

        $transfer_cat_id = Category::transfer();

        return view('accounts.expenses.payments.index', compact('payments', 'vendors', 'categories', 'accounts', 'transfer_cat_id'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        $accounts = Account::whereEnabled(1)->get()->pluck('name_number', 'id');

        $currencies = Currency::whereEnabled(1)->pluck('name', 'code')->toArray();

        $account_currency_code = Account::where('id', setting('general.default_account'))->pluck('currency_code')->first();

        $vendors = Vendor::whereEnabled(1)->pluck('name', 'id');

        $categories = Category::whereEnabled(1)->type('expense')->pluck('name', 'id');

        $payment_methods = Modules::getPaymentMethods();

          $terms = Terms::pluck('name','id');

        return view('accounts.expenses.payments.create', compact('accounts', 'currencies', 'account_currency_code', 'vendors', 'categories', 'payment_methods','terms'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  Request  $request
     *
     * @return Response
     */
    public function store(Request $request)
    {
        // Get currency object
        $currency = Currency::where('code', 'USD')->first();
        $request['currency_code'] = $currency->code;
        $request['currency_rate'] = $currency->rate;
        $pdate = $this->date_format_change_unique($request->paid_at)[0];
        $request['paid_at'] =  date("Y-m-d", strtotime($pdate));

        $payment = Payment::create($request->input());
        // Upload attachment
        $media = $this->getMedia($request->file('attachment'), 'payments');

        if ($media) {
            $payment->attachMedia($media, 'attachment');
        }

        $message = trans('messages.success.added', ['type' => trans_choice('general.payments', 1)]);

        Session::flash('success',$message);

        return redirect('expenses/payments');
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param  Payment  $payment
     *
     * @return Response
     */
    public function edit(Payment $payment)
    {
        $accounts = Account::whereEnabled(1)->get()->pluck('name_number', 'id');

        $currencies = Currency::whereEnabled(1)->pluck('name', 'code')->toArray();

        $account_currency_code = Account::where('id', $payment->account_id)->pluck('currency_code')->first();

        $vendors = Vendor::whereEnabled(1)->pluck('name', 'id');

        $categories = Category::whereEnabled(1)->type('expense')->pluck('name', 'id');

        $payment_methods = Modules::getPaymentMethods();

        return view('accounts.expenses.payments.edit', compact('payment', 'accounts', 'currencies', 'account_currency_code', 'vendors', 'categories', 'payment_methods'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  Payment  $payment
     * @param  Request  $request
     *
     * @return Response
     */
    public function update(Payment $payment, Request $request)
    {
        // Get currency object
        $currency = Currency::where('code', $request['currency_code'])->first();

        $request['currency_code'] = $currency->code;
        $request['currency_rate'] = $currency->rate;

        $payment->update($request->input());

        // Upload attachment
        if ($request->file('attachment')) {
            $media = $this->getMedia($request->file('attachment'), 'payments');

            $payment->attachMedia($media, 'attachment');
        }

        $message = trans('messages.success.updated', ['type' => trans_choice('general.payments', 1)]);

        Session::flash('success',$message);

        return redirect('expenses/payments');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  Payment  $payment
     *
     * @return Response
     */
    public function destroy(Payment $payment)
    {
        // Can't delete transfer payment
        if ($payment->category->id == Category::transfer()) {
            return redirect('expenses/payments');
        }

        $payment->delete();

        $message = trans('messages.success.deleted', ['type' => trans_choice('general.payments', 1)]);

        Session::flash('success',$message);

        return redirect('expenses/payments');
    }
}
