<?php

namespace App\Http\Controllers\Accounts\Expenses;

use Image;
use File;
use Date;
use App\Traits\Uploads;
use App\Traits\DateTime;
use App\Traits\Currencies;
use App\Utilities\Modules;
use App\Classes\DepositType;
use Illuminate\Http\Request;
use App\Models\Accounts\Item\Item;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\Accounts\Common\Media;
use App\Models\Accounts\Charthistory;
use App\Models\Accounts\Income\Terms;
use App\Models\Accounts\Expense\Cheque;
use App\Models\Accounts\Expense\Vendor;
use App\Models\Accounts\Banking\Account;
use App\Models\Accounts\Income\Customer;
use App\Models\Accounts\Setting\Category;
use App\Models\Accounts\Setting\Currency;
use App\Models\Accounts\CharthistoryParent;
use App\Models\Accounts\Expense\ChequeTotal;
use App\Models\Accounts\Expense\ChequeAccount;
use App\Models\Accounts\Expense\ChequeHistory;
use App\Models\Accounts\Accounting\Chartofaccounts;

class ChequeController extends Controller
{
    use DateTime, Currencies, Uploads;
    public $personname = [];

    public function index()
    {
        $cheques = Cheque::with([ 'status', 'items', 'payments', 'histories'])->orderBy('created_at','desc')->get()->collect();        

        return view('accounts.expenses.cheques.index', compact('cheques'));
    }

    public function create($id='')
    { 
        $this->addRecevedFrom(Customer::select(DB::raw('CONCAT(id,"-",1) as idconcate'),'name')->get(), 'Customer');
        $this->addRecevedFrom(Vendor::select(DB::raw('CONCAT(id,"-",2) as idconcate'),'name')->get(), 'Vendor');

        $personname = $this->getPersonname();

        $currencies = Currency::whereEnabled(1)->pluck('name', 'code');
        $accounts = Account::whereEnabled(1)->get()->pluck('name_number', 'id');
        $items = Item::whereEnabled(1)->pluck('name', 'id');
        $number = $this->getNextChequeNumber();
        $terms = Terms::pluck('name','id');

        $vendor = isset($id)?$id.'-2' :0;
       
        return view('accounts.expenses.cheques.create', compact('personname','currencies','accounts','items','number','vendor','terms'));
    }

    protected function addRecevedFrom($items,$type)
    {   
        foreach ($items as $item) {
           $data = [ 
            'id' => $item->idconcate,
            'name' => $item->name,
            'type' => $type
           ];
        $this->personname[] = (object) $data;
        }
    }

    protected function getPersonname()
    {
        $personname = collect($this->personname);
       
        return $personname;
    }

    public static function getNextChequeNumber()
    {
        $prefix = setting('general.cheque_number_prefix', 'CHE-');
        $next = setting('general.cheque_number_next', '1');
        $digit = setting('general.cheque_number_digit', '5');

        $number = $prefix . str_pad($next, $digit, '0', STR_PAD_LEFT);

        return $number;
    }

    public function store(Request $request)
    {
        // dd($request->all());
         // Update next invoice number
        $this->increaseNextChequeNumber();
        if($request['person_id']){   
            $person_find = explode("-",$request['person_id'])[1];
            if($person_find === 1){
                $customer = Customer::find(explode("-",$request['person_id'])[0]);
                $request['person_name'] = isset($customer->name)?$customer->name:'';
                $request['person_email'] = isset($customer->email)?$customer->email:'';
                $request['person_phone'] = isset($customer->phone)?$customer->phone:'';
                $request['person_address'] = isset($customer->address)?$customer->address:'';
                $request['person_type'] = 'customer';
            }else{ 
                $vendor = Vendor::find(explode("-",$request['person_id'])[0]);
                $request['person_name'] = isset($vendor->name)?$vendor->name:'';
                $request['person_email'] = isset($vendor->email)?$vendor->email:'';
                $request['person_phone'] = isset($vendor->phone)?$vendor->phone:'';
                $request['person_address'] = isset($vendor->address)?$vendor->address:'';
                $request['person_type'] = 'vendor';
            } 
        }

        $request['currency_code']  = 'USD';
        $request['company_id']  = 1;
        // Get currency object
        $currency = Currency::where('code', $request['currency_code'])->first();

        $request['currency_code'] = $currency->code;
        $request['currency_rate'] = $currency->rate;
        $date = $this->date_format_change_unique($request->paid_at , $request->paid_at);
        
        $request['paid_at']    =  date("Y-m-d", strtotime($date[0]));
        $request['due_at']    =  date("Y-m-d", strtotime($date[1]));
        
        $request['cheque_status_code'] = 'draft';

        $request['amount'] = 0;

        $cheque = Cheque::create($request->input());

        // Upload attachment
        if ($request->file('attachment')) {
           
            $media = $this->getMedia($request->file('attachment'), 'cheques');
          
            $cheque->attachMedia($media, 'attachment');
        }
        $sub_total = 0;

        $cheque_item = [];
        $cheque_item['company_id'] = 1;
        $cheque_item['cheque_id'] = $cheque->id;

        //Set Chart History.
        $chart_history = [];
        $chart_history['company_id'] = 1;
        $chart_history['transaction_id'] = $cheque->id;
        $chart_history['transaction_type'] = 'Cheques';
        $chart_history['number'] = $cheque->cheque_number;
        $chart_history['currency_code'] = $currency->code;
        $chart_history['name'] = $request['person_name'];        

        $cheque_account = [];
        $cheque_account['company_id'] = 1;
        $cheque_account['cheque_id'] = $cheque->id;

        if($request['account'] && $request['account'][0]['name']){
            foreach ($request['account'] as $account) {
                $chart_object = Chartofaccounts::find($account['account_id']);
               // dd($account['account_id']);
                $cheque_account['account_type_id'] = $account['account_id'];
                $cheque_account['quantity'] = $account['quantity'];
                $cheque_account['name'] = \Str::limit($account['name'], 180, '');
                $cheque_account['price'] = $account['price'];
                $cheque_account['description'] = $account['description'];
                $cheque_account['company_id'] = 1;
                $cheque_account['total'] = $account['price'] * $account['quantity'];
                ChequeAccount::create($cheque_account);

                // Set Chart History.
                $chart_history['amount_type'] = DepositType::find($account['account_id'],'debit');
                $chart_history['chartofaccount_id'] = $account['account_id'];
                $chart_history['account_type_id'] = $chart_object['account_type_id'];
                $chart_history['detail_type'] = $chart_object['detail_type'];
                $chart_history['amount'] = $account['price'] * $account['quantity'];
                $chart_history['description'] = $request->description;
                $chart_history['company_id'] = 1;
                $chart_history['transaction_date'] = $this->formatDate($request->paid_at);

                $chart_history_id = Charthistory::create($chart_history);

                $chartparents = app('App\Http\Controllers\Accounts\Accounting\Accounting')->parent_chartaccount_id($account['account_id']);
 
                CharthistoryParent::charthistoryparentsave($chart_history_id->id,$chartparents->original['parent_id_string'],$account['account_id']); 

                $sub_total += $account['price'] * $account['quantity'];
            }
        }  

        $request['amount'] = $sub_total;
        $cheque->update($request->input());
        $chart_object = Chartofaccounts::where('bank_id', $request->account_id)->first();

        $chart_history_bank = [];
        $chart_history_bank['company_id'] = 1;
        $chart_history_bank['transaction_id'] = $cheque->id;
        $chart_history_bank['transaction_type'] = 'Cheques';
        $chart_history_bank['number'] = $cheque->cheque_number;
        $chart_history_bank['currency_code'] = $currency->code;
        $chart_history_bank['name'] = $request['person_name'];
        $chart_history_bank['amount_type'] = DepositType::find($chart_object->id,'credit');
        $chart_history_bank['chartofaccount_id'] = $chart_object->id;
        $chart_history_bank['account_type_id'] = $chart_object->account_type_id;
        $chart_history_bank['detail_type'] = $chart_object->detail_type;
        $chart_history_bank['amount'] = $sub_total;
        $chart_history_bank['description'] = $request->description;
        $chart_history_bank['transaction_date'] = $request['due_at'];

        $chart_history_id = Charthistory::create($chart_history_bank);

        $chartparents = app('App\Http\Controllers\Accounts\Accounting\Accounting')->parent_chartaccount_id($chart_object->id);

        CharthistoryParent::charthistoryparentsave($chart_history_id->id,$chartparents->original['parent_id_string'],$chart_object->id); 

        // Add cheque totals
        $this->addTotals($cheque, $request, $sub_total, 0);

        // Add cheque history
        ChequeHistory::create([
            'company_id' => 1,
            'cheque_id' => $cheque->id,
            'status_code' => 'draft',
            'notify' => 0,
            'description' => trans('messages.success.added', ['type' => $cheque->cheque_number]),
        ]);
        
        return redirect('expenses/cheques/' . $cheque->id);
    }

    public function increaseNextChequeNumber()
    {
        // Update next cheque number
        $next = setting('general.cheque_number_next', 1) + 1;
        setting(['general.cheque_number_next' => $next]);
        setting()->save();
    }

    protected function addTotals($cheque, $request, $sub_total, $tax_total)
    {
        $sort_order = 1;
        // Added cheque total sub total
        ChequeTotal::create([
            'company_id' => $request['company_id'],
            'cheque_id' => $cheque->id,
            'code' => 'sub_total',
            'name' => 'cheques.sub_total',
            'amount' => $sub_total,
            'sort_order' => $sort_order,
        ]);

        $sort_order++;
        
        // Added cheque total total
        ChequeTotal::create([
            'company_id' => $request['company_id'],
            'cheque_id' => $cheque->id,
            'code' => 'total',
            'name' => 'cheques.total',
            'amount' => $sub_total,
            'sort_order' => $sort_order,
        ]);
    }

    public function show(Cheque $cheque)
    {
        $paid = 0;

        foreach ($cheque->payments as $item) {
            $item->default_currency_code = $cheque->currency_code;

            $paid += $item->getDynamicConvertedAmount();
        }

        $cheque->paid = $paid;
        $accounts = Account::whereEnabled(1)->get()->pluck('name_number', 'id');

        $currencies = Currency::pluck('name', 'code')->toArray();

        $account_currency_code = Account::where('id', setting('general.default_account'))->pluck('currency_code')->first();

        $vendors = Vendor::pluck('name', 'id');

        $categories = Category::type('income')->pluck('name', 'id');

        $payment_methods = Modules::getPaymentMethods();

        return view('accounts.expenses.cheques.show', compact('cheque', 'accounts', 'currencies', 'account_currency_code', 'vendors', 'categories', 'payment_methods'));        
    }

    public function edit(Cheque $cheque)
    {
       $this->addRecevedFrom(Customer::select(\DB::raw('CONCAT(id,"-",1) as idconcate'),'name')->get(),trans_choice('general.customers', 1));
        $this->addRecevedFrom(Vendor::select(\DB::raw('CONCAT(id,"-",2) as idconcate'),'name')->get(),trans_choice('general.vendors', 1));

        $personname = $this->getPersonname();

        $currencies = Currency::pluck('name', 'code')->all();

        $accounts = Account::get()->pluck('name_number', 'id');

        return view('accounts.expenses.cheques.edit', compact('cheque', 'personname', 'currencies','accounts'));
    }

    public function update(Cheque $cheque, Request $request)
    {
       if ($request['person_id']) {
            $person_find = explode("-",$request['person_id'])[1];
           
            if($person_find === "1"){
                $customer = Customer::find(explode("-",$request['person_id'])[0]);
                $request['person_name'] = $customer->name;
                $request['person_email'] = $customer->email;
                $request['person_phone'] = $customer->phone;
                $request['person_address'] = $customer->address;
                $request['person_type'] = 'customer';

            }else{ 
                $vendor = Vendor::find(explode("-",$request['person_id'])[0]);
                $request['person_name'] = $vendor->name;
                $request['person_email'] = $vendor->email;
                $request['person_phone'] = $vendor->phone;
                $request['person_address'] = $vendor->address;
                $request['person_type'] = 'vendor';
            }
       } 

        // Get currency object
        $currency = Currency::where('code', $request['currency_code'])->first();
        
        $date = $this->date_format_change_unique($request->paid_at);

        $request['paid_at'] = date("Y-m-d",strtotime($date[0]));
        //$request['due_at'] = date("Y-m-d",strtotime($date[1]));

        $request['currency_code'] = $currency->code;
        $request['currency_rate'] = $currency->rate;

        $sub_total = 0;
        
        $cheque_account = [];
        $cheque_account['company_id'] = session('company_id');
        $cheque_account['cheque_id'] = $cheque->id;
       
        $chart_history = [];
        $chart_history['company_id'] = session('company_id');
        $chart_history['transaction_id'] = $cheque->id;
        $chart_history['transaction_type'] = 'Cheques';
        $chart_history['number'] = $cheque->cheque_number;
        $chart_history['currency_code'] = $currency->code;
        $chart_history['name'] =  $request['person_name'];

        ChequeAccount::where('cheque_id', $cheque->id)->delete();
        if($request['account']){
            foreach ($request['account'] as $account) {
                Charthistory::where('transaction_type', 'Cheques')->where('transaction_id',$cheque->id)->delete();
                $chart_object = Chartofaccounts::find($account['account_id']);               
               
                $cheque_account['account_type_id'] = $account['account_id'];
                $cheque_account['quantity'] = $account['quantity'];
                $cheque_account['name'] = \Str::limit($chart_object['name'], 180, '');
                $cheque_account['price'] = $account['price'];
                $cheque_account['company_id'] = session('company_id');
                $cheque_account['description'] = $account['description'];
                $cheque_account['total'] = $account['price'] * $account['quantity'];
                ChequeAccount::create($cheque_account);

                // Set Chart History.
                $chart_history['amount_type'] = DepositType::find($account['account_id'],'debit');
                $chart_history['chartofaccount_id'] = $account['account_id'];
                $chart_history['account_type_id'] = $chart_object['account_type_id'];
                $chart_history['detail_type'] = $chart_object['detail_type'];
                $chart_history['amount'] = $account['price'] * $account['quantity'];
                $chart_history['description'] = $account['description'];
                $chart_history['company_id'] = session('company_id');
                $chart_history['transaction_date'] = $this->formatDate($request->paid_at);
                $chart_history_id = Charthistory::create($chart_history);

                $chartparents = app('App\Http\Controllers\Accounts\Accounting\Accounting')->parent_chartaccount_id($account['account_id']);

                CharthistoryParent::charthistoryparentsave($chart_history_id->id,$chartparents->original['parent_id_string'],$account['account_id']); 
                 

                $sub_total += $cheque_account['total'];

            }
        }  

        $request['amount'] = $sub_total;

        $cheque->update($request->input());

        $chart_object = Chartofaccounts::where('bank_id',$request->account_id)->first();

        $chart_history_bank = [];
        $chart_history_bank['company_id'] = session('company_id');
        $chart_history_bank['transaction_id'] = $cheque->id;
        $chart_history_bank['transaction_type'] = 'Cheques';
        $chart_history_bank['number'] = $cheque->cheque_number;
        $chart_history_bank['currency_code'] = $currency->code;
        $chart_history_bank['name'] = $request['person_name'];
        $chart_history_bank['amount_type'] = DepositType::find($chart_object->id,'credit');
        $chart_history_bank['chartofaccount_id'] = $chart_object->id;
        $chart_history_bank['account_type_id'] = $chart_object->account_type_id;
        $chart_history_bank['detail_type'] = $chart_object->detail_type;
        $chart_history_bank['amount'] = $sub_total;
        $chart_history_bank['description'] = $request->description;
        $chart_history_bank['transaction_date'] = $this->formatDate($request->paid_at);

        $chart_history_id = Charthistory::create($chart_history_bank);

        $chartparents = app('App\Http\Controllers\Accounts\Accounting\Accounting')->parent_chartaccount_id($chart_object->id);

        CharthistoryParent::charthistoryparentsave($chart_history_id->id,$chartparents->original['parent_id_string'],$chart_object->id); 

        // Upload attachment
        if ($request->file('attachment')) {
            
            $media = $this->getMedia($request->file('attachment'), 'cheques');
            $cheque->attachMedia($media, 'attachment');
        }

        // Delete previous cheque totals
        ChequeTotal::where('cheque_id', $cheque->id)->delete();

        // Add cheque totals
        $this->addTotals($cheque, $request, $sub_total, 0);
        
        return redirect('expenses/cheques/' . $cheque->id);
    }

    public function printCheque(Cheque $cheque)
    {
        $cheque = $this->prepareCheque($cheque);
        $logo = $this->getLogo($cheque);

        return view($cheque->template_path, compact('cheque', 'logo'));
    }

    protected function prepareCheque(Cheque $cheque)
    {
        $paid = 0;

        foreach ($cheque->payments as $item) {
            $item->default_currency_code = $cheque->currency_code;

            $paid += $item->getDynamicConvertedAmount();
        }

        $cheque->paid = $paid;

        $cheque->template_path = 'accounts.expenses.cheques.cheque';
        
        return $cheque;
    }

    protected function getLogo($cheque)
    {
        $logo = '';

        $media_id = setting('general.company_logo');

        if (isset($cheque->vendor->logo) && !empty($cheque->vendor->logo->id)) {
            $media_id = $cheque->vendor->logo->id;
        }

        $media = Media::find($media_id);

         if (!empty($media)) {
            $path = storage_path( 'app/uploads/' . $media->getDiskPath()); 
            if (!is_file($path)) {
                return $logo;
            }
        } else {
            $path = asset('public/accounts/img/company.png');
        }

        $image = Image::make($path)->encode()->getEncoded();

        if (empty($image)) {
            return $logo;
        }

        $extension = File::extension($path);

        $logo = 'data:image/' . $extension . ';base64,' . base64_encode($image);

        return $logo;
    }
}
