<?php

namespace App\Http\Controllers\Accounts\Expenses;

use App\Traits\Uploads;
use App\Traits\DateTime;
use App\Traits\Currencies;
use App\Utilities\Modules;
use App\Events\BillCreated;
use App\Events\BillUpdated;
use Illuminate\Support\Str;
use App\Classes\DepositType;
use Illuminate\Http\Request;
use App\Utilities\ImportFile;
use App\Models\Accounts\Item\Item;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\Accounts\Charthistory;
use App\Models\Accounts\Expense\Bill;
use App\Models\Accounts\Income\Terms;
use App\Models\Accounts\Expense\Vendor;
use Illuminate\Support\Facades\Session;
use App\Models\Accounts\Banking\Account;
use App\Models\Accounts\Setting\Category;
use App\Models\Accounts\Setting\Currency;
use App\Models\Accounts\Expense\BillTotal;
use App\Models\Accounts\CharthistoryParent;
use App\Models\Accounts\Expense\BillStatus;
use App\Models\Accounts\Expense\BillAccount;
use App\Models\Accounts\Expense\BillHistory;
use App\Models\Accounts\Expense\BillPayment;
use App\Models\Accounts\Accounting\Chartofaccounts;
use App\Http\Requests\Expense\BillPayment as PaymentRequest;

class Bills extends Controller
{
    use DateTime, Currencies, Uploads;

    /**
     * Display a listing of the resource.
     *
     * @return Response
     */
    public function index()
    {
        $bills = Bill::with(['vendor', 'status', 'items', 'payments', 'histories'])->get()->collect(['billed_at'=> 'desc']);
        $vendors = collect(Vendor::pluck('name', 'id'))
            ->prepend(trans('general.all_type', ['type' => trans_choice('general.vendors', 2)]), '');

        $statuses = collect(BillStatus::all()->pluck('name', 'code'))
            ->prepend(trans('general.all_type', ['type' => trans_choice('general.statuses', 2)]), '');

        return view('accounts.expenses.bills.index', compact('bills', 'vendors', 'statuses'));
    }

    /**
     * Show the form for viewing the specified resource.
     *
     * @param  Bill  $bill
     *
     * @return Response
     */
    public function show(Bill $bill)
    {
        $paid = 0;

        foreach ($bill->payments as $item) {
            $item->default_currency_code = $bill->currency_code;

            $paid += $item->getDynamicConvertedAmount();
        }

        $bill->paid = $paid;

        $accounts = Account::whereEnabled(1)->get()->pluck('name_number', 'id');

        $currencies = Currency::whereEnabled(1)->pluck('name', 'code')->toArray();

        $account_currency_code = Account::where('id', setting('general.default_account'))->pluck('currency_code')->first();

        $vendors = Vendor::whereEnabled(1)->pluck('name', 'id');

        $categories = Category::whereEnabled(1)->type('income')->pluck('name', 'id');

        $payment_methods = Modules::getPaymentMethods();

        return view('accounts.expenses.bills.show', compact('bill', 'accounts', 'currencies', 'account_currency_code', 'vendors', 'categories', 'payment_methods'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create($id='')
    {
        $vendors = Vendor::whereEnabled(1)->pluck('name', 'id');

        $currencies = Currency::whereEnabled(1)->pluck('name', 'code');

        $elements = Item::whereEnabled(1)->select('name', 'id','subitem')->get()->toArray();

        $items  = $this->bulidItems($elements);

        $terms = Terms::pluck('name','id');

        $vendor = isset($id)?$id:0;

        return view('accounts.expenses.bills.create', compact('vendors','currencies', 'items', 'vendor','terms'));
    }

    function bulidItems(array $elements, $parentId = NULL) {
            $child = array();
        
            foreach ($elements as $element) {
                if ($element['subitem'] == $parentId) {

                    $children = $this->bulidItems($elements, $element['id']);

                        if (!empty($children) > 0 ) {
                            $element['children'] = $children;
                        }

                    $child[] = $element;
                }
            }
        return $child;
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  Request  $request
     *
     * @return Response
     */
    public function store(Request $request)
    {
        DB::beginTransaction();

       // Get vendor object
        $vendor = Vendor::find($request['vendor_id']);

        $request['vendor_name'] = $vendor->name;
        $request['vendor_email'] = $vendor->email;
        $request['vendor_tax_number'] = $vendor->tax_number;
        $request['vendor_phone'] = $vendor->phone;
        $request['vendor_address'] = $vendor->address;

        // Get currency object
        $currency = Currency::where('code', 'USD')->first();

        $request['currency_code'] = $currency->code;
        $request['currency_rate'] = $currency->rate;
        
        $request['bill_status_code'] = 'draft';
        $date = $this->date_format_change_unique($request->billed_at,$request->due_at);
        $request['billed_at'] =  date("Y-m-d", strtotime($date[0])); 
        $request['due_at'] = date("Y-m-d", strtotime($date[1])); 
        $request['amount'] = 0;
        
        $bill = Bill::create($request->input());

        // Upload attachment
        if ($request->file('attachment')) {
            $media = $this->getMedia($request->file('attachment'), 'bills');
            $bill->attachMedia($media, 'attachment');
        }

        $sub_total = 0;

        $bill_item = [];
        $bill_item['company_id'] = session('company_id');
        $bill_item['bill_id'] = $bill->id;

        //Set Chart History.
        $chart_history = [];
        $chart_history['company_id'] = session('company_id');
        $chart_history['transaction_id'] = $bill->id;
        $chart_history['transaction_type'] = 'Bills';
        $chart_history['number'] = $bill->bill_number;
        $chart_history['currency_code'] = $currency->code;
        $chart_history['name'] = $vendor->name;               
        $chart_history['transaction_date'] = date("Y-m-d", strtotime($date[0]));                

        $bill_account = [];
        $bill_account['company_id'] = $request['company_id'];
        $bill_account['bill_id'] = $bill->id;

        if($request['account']){
            foreach ($request['account'] as $account) {
                $chart_object = Chartofaccounts::find($account['account_id']);

                $bill_account['account_type_id'] = $account['account_id'];
                $bill_account['quantity'] = $account['quantity'];
                $bill_account['name'] = Str::limit($account['name'], 180, '');
                $bill_account['price'] = $account['price'];
                $bill_account['description'] = $account['description'];
                $bill_account['total'] = $account['quantity'] * $account['price'];
                BillAccount::create($bill_account);

                // Set Chart History.
                $chart_history['amount_type'] = DepositType::find($account['account_id'], 'debit'); 
                $chart_history['chartofaccount_id'] = $account['account_id'];
                $chart_history['account_type_id'] = $chart_object['account_type_id'];
                $chart_history['detail_type'] = $chart_object['detail_type'];
                $chart_history['amount'] = $account['price'];
                $chart_history['description'] = $account['description'];
                $chart_history_id = Charthistory::create($chart_history);

                $chartparents = app('App\Http\Controllers\Accounts\Accounting\Accounting')->parent_chartaccount_id($account['account_id']);
 
                CharthistoryParent::charthistoryparentsave($chart_history_id->id,$chartparents->original['parent_id_string'],$account['account_id']);

                // Calculate totals
                $sub_total += $bill_account['total'];
            }
        } 

        // Updating Amount for Bill
        $request['amount'] += $sub_total;
        $bill->update($request->input());

        //History For Account Payable
        $chart =  Chartofaccounts::where('name','=','Accounts Payable')->first();
        $chart_receivable = [];
        $chart_receivable['company_id'] = session('company_id');
        $chart_receivable['transaction_id'] = $bill->id;
        $chart_receivable['transaction_type'] = 'Bills';
        $chart_receivable['number'] = $bill->bill_number;
        $chart_receivable['currency_code'] = $currency->code;
        $chart_receivable['name'] = $vendor->name;
        $chart_receivable['amount_type'] = DepositType::find($chart->id, 'credit');
        $chart_receivable['chartofaccount_id'] = $chart->id;
        $chart_receivable['account_type_id'] = $chart->account_type_id;
        $chart_receivable['detail_type'] = $chart->detail_type;
        $chart_receivable['amount'] = $sub_total;
        $chart_receivable['description'] = $request->notes;
        $chart_receivable['transaction_date'] = date('Y-m-d',strtotime($date[0]));



        $chart_receivable_id = Charthistory::create($chart_receivable);

        $chartparents = app('App\Http\Controllers\Accounts\Accounting\Accounting')->parent_chartaccount_id($chart->id);
 
        CharthistoryParent::charthistoryparentsave($chart_receivable_id->id,$chartparents->original['parent_id_string'],$chart->id);

        // Add bill totals
        $this->addTotals($bill, $request, $sub_total);

        // Add bill history
        BillHistory::create([
            'company_id' => session('company_id'),
            'bill_id' => $bill->id,
            'status_code' => 'draft',
            'notify' => 0,
            'description' => trans('messages.success.added', ['type' => $bill->bill_number]),
        ]);

        DB::commit();

        // Fire the event to make it extendible
        event(new BillCreated($bill));

        $message = trans('messages.success.added', ['type' => trans_choice('general.bills', 1)]);

        return redirect('expenses/bills/' . $bill->id)->with('success', $message);
    }

    /**
     * Import the specified resource.
     *
     * @param  ImportFile  $import
     *
     * @return Response
     */
    public function import(ImportFile $import)
    {
        $rows = $import->all();

        foreach ($rows as $row) {
            $data = $row->toArray();
            $data['company_id'] = session('company_id');

            Bill::create($data);
        }

        $message = trans('messages.success.imported', ['type' => trans_choice('general.bills', 2)]);

        Session::flash('success',$message);

        return redirect('expenses/bills');
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  Bill  $bill
     *
     * @return Response
     */
    public function edit(Bill $bill)
    {   
        $vendors = Vendor::whereEnabled(1)->pluck('name', 'id');

        $currencies = Currency::whereEnabled(1)->pluck('name', 'code');

        $items = Item::whereEnabled(1)->pluck('name', 'id');

        $terms = Terms::pluck('name','id');
      
        return view('accounts.expenses.bills.edit', compact('bill', 'vendors', 'currencies', 'items', 'terms'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  Bill  $bill
     * @param  Request  $request
     *
     * @return Response
     */
    public function update(Bill $bill, Request $request)
    {
        DB::beginTransaction();
        
        // Deleting previous bills
        Charthistory::where('transaction_id',$bill->id)->whereIn('transaction_type',['Bills','Bill (Payment)'])->delete();
  
        // Get vendor object
        $vendor = Vendor::find($request['vendor_id']);

        $request['vendor_name'] = $vendor->name;
        $request['vendor_email'] = $vendor->email;
        $request['vendor_tax_number'] = $vendor->tax_number;
        $request['vendor_phone'] = $vendor->phone;
        $request['vendor_address'] = $vendor->address;

        // Get currency object
        $currency = Currency::where('code', 'USD')->first();
        $date = $this->date_format_change_unique($request->billed_at, $request->due_at);
        $request['billed_at'] =  date("Y-m-d", strtotime($date[0])); 
        $request['due_at'] = date("Y-m-d", strtotime($date[1])); 
        
        $request['currency_code'] = $currency->code;
        $request['currency_rate'] = $currency->rate;

        $sub_total = 0;

        $bill_item = [];
        $bill_item['company_id'] = $request['company_id'];
        $bill_item['bill_id'] = $bill->id;

        //Set Chart History.
        $chart_history = [];
        $chart_history['company_id'] = session('company_id');
        $chart_history['transaction_id'] = $bill->id;
        $chart_history['transaction_type'] = 'Bills';
        $chart_history['number'] = $bill->bill_number;
        $chart_history['currency_code'] = $currency->code;
        $chart_history['name'] = $vendor->name;
        $chart_history['transaction_date'] = date("Y-m-d", strtotime($date[0]));   

        $bill_account = [];
        $bill_account['company_id'] = $request['company_id'];
        $bill_account['bill_id'] = $bill->id;

        if($request['account']){    
            // Deleting previous bill accounts   
            BillAccount::where('bill_id', $bill->id)->delete();

            foreach ($request['account'] as $account) {
                $chart_object = Chartofaccounts::find($account['account_id']);

                $bill_account['account_type_id'] = $account['account_id'];
                $bill_account['quantity'] = $account['quantity'];
                $bill_account['name'] = Str::limit($account['name'], 180, '');
                $bill_account['price'] = $account['price'];
                $bill_account['description'] = $account['description'];
                $bill_account['total'] = $account['price'] * $account['quantity'];
                BillAccount::create($bill_account);

                // Set Chart History.
                $chart_history['amount_type'] = DepositType::find($account['account_id'], 'debit'); 
                $chart_history['chartofaccount_id'] = $account['account_id'];
                $chart_history['account_type_id'] = $chart_object['account_type_id'];
                $chart_history['detail_type'] = $chart_object['detail_type'];
                $chart_history['amount'] = $account['price'];
                $chart_history['description'] = $account['description'];
                $chart_history_id = Charthistory::create($chart_history);

                $chartparents = app('App\Http\Controllers\Accounts\Accounting\Accounting')->parent_chartaccount_id($account['account_id']);
 
                CharthistoryParent::charthistoryparentsave($chart_history_id->id,$chartparents->original['parent_id_string'],$account['account_id']);

                $sub_total += $bill_account['total'];
            }
        }  

        // Updating Amount for Bill
        $request['amount'] += $sub_total;
        $bill->update($request->input());
        // Upload attachment for Bill
        if ($request->file('attachment')) {
            $media = $this->getMedia($request->file('attachment'), 'bills');
            $bill->attachMedia($media, 'attachment');
        }

        //History For Account Payable
        $chart =  Chartofaccounts::where('name','=','Accounts Payable')->first();
        $chart_receivable = [];
        $chart_receivable['company_id'] = session('company_id');
        $chart_receivable['transaction_id'] = $bill->id;
        $chart_receivable['transaction_type'] = 'Bills';
        $chart_receivable['number'] = $bill->bill_number;
        $chart_receivable['currency_code'] = $currency->code;
        $chart_receivable['name'] = $vendor->name;
        $chart_receivable['amount_type'] = DepositType::find($chart->id, 'credit');
        $chart_receivable['chartofaccount_id'] = $chart->id;
        $chart_receivable['account_type_id'] = $chart->account_type_id;
        $chart_receivable['detail_type'] = $chart->detail_type;
        $chart_receivable['amount'] = $sub_total;
        $chart_receivable['description'] = $request->notes;
        $chart_receivable['transaction_date'] = date("Y-m-d", strtotime($date[0]));   
        $chart_receivable_id = Charthistory::create($chart_receivable);       


        $chartparents = app('App\Http\Controllers\Accounts\Accounting\Accounting')->parent_chartaccount_id($chart->id);
 
        CharthistoryParent::charthistoryparentsave($chart_receivable_id->id,$chartparents->original['parent_id_string'],$chart->id); 

        // Delete previous bill totals by adding at store method using addTotals()
        BillTotal::where('bill_id', $bill->id)->delete();

        // Add bill totals
        $this->addTotals($bill, $request, $sub_total);

        DB::commit();

        // Fire the event to make it extendible
        event(new BillUpdated($bill));

        $message = trans('messages.success.updated', ['type' => trans_choice('general.bills', 1)]);

        return redirect('expenses/bills/' . $bill->id)->with('success', $message);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  Bill  $bill
     *
     * @return Response
     */
    public function destroy(Bill $bill)
    {
        $bill->delete();

        /*
        $bill->items->delete();
        $bill->payments->delete();
        $bill->histories->delete();
        */

        BillItem::where('bill_id', $bill->id)->delete();
        BillAccount::where('bill_id', $bill->id)->delete();
        BillTotal::where('bill_id', $bill->id)->delete();
        BillPayment::where('bill_id', $bill->id)->delete();
        BillHistory::where('bill_id', $bill->id)->delete();

        $message = trans('messages.success.deleted', ['type' => trans_choice('general.bills', 1)]);

        \Session::flash('success',$message);

        return redirect('expenses/bills');
    }

    /**
     * Mark the bill as received.
     *
     * @param  Bill $bill
     *
     * @return Response
     */
    public function markReceived(Bill $bill)
    {
        $bill->bill_status_code = 'received';
        $bill->save();

        flash(trans('bills.messages.received'))->success();

        return redirect()->back();
    }

    /**
     * Print the bill.
     *
     * @param  Bill $bill
     *
     * @return Response
     */
    public function printBill(Bill $bill)
    {
        $bill = $this->prepareBill($bill);

        $logo = $this->getLogo($bill);

        return view($bill->template_path, compact('bill', 'logo'));
    }

    /**
     * Download the PDF file of bill.
     *
     * @param  Bill $bill
     *
     * @return Response
     */
    public function pdfBill(Bill $bill)
    {

        $bill = $this->prepareBill($bill);

        $logo = $this->getLogo($bill);

        $html = view($bill->template_path, compact('bill', 'logo'))->render();

        $pdf = \App::make('dompdf.wrapper');
        $pdf->loadHTML($html);

        $file_name = 'bill_' . time() . '.pdf';

        return $pdf->download($file_name);
    }

    /**
     * Add payment to the bill.
     *
     * @param  PaymentRequest  $request
     *
     * @return Response
     */
    public function payment(PaymentRequest $request)
    {
        DB::beginTransaction();

        // Get currency object
        $currency = Currency::where('code', 'USD')->first();

        $request['currency_code']   = $currency->code;
        $request['currency_rate']   = $currency->rate;
        $request['company_id']      = session('company_id');
        $request['paid_at']         = $this->formatDate($request->paid_at);

        $bill = Bill::find($request['bill_id']);

        $total_amount = $bill->amount;

        $amount = (double) $request['amount'];

        if ($request['currency_code'] != $bill->currency_code) {
            $request_bill = new Bill();

            $request_bill->amount = (float) $request['amount'];
            $request_bill->currency_code = $currency->code;
            $request_bill->currency_rate = $currency->rate;

            $amount = $request_bill->getConvertedAmount();
        }

        if ($bill->payments()->count()) {
            $total_amount -= $bill->payments()->paid();
        }

        if ($amount > $total_amount) {
            $message = trans('messages.error.over_payment');
            DB::rollback();
            return response()->json([
                'success' => false,
                'error' => true,
                'message' => $message,
            ]);
        } elseif ($amount == $total_amount) {
            $bill->bill_status_code = 'paid';
        } else {
            $bill->bill_status_code = 'partial';
        }

        $bill->save();

        //Set Chart History.
        $chart_object = Chartofaccounts::find($request->account_chart_id);

        $chart_history = new Charthistory();
        $chart_history->company_id = session('company_id');
        $chart_history->chartofaccount_id = $request->account_chart_id;
        $chart_history->transaction_type = 'Bill (Payment)';
        $chart_history->currency_code = $currency->code;
        $chart_history->account_type_id = $chart_object->account_type_id;
        $chart_history->detail_type = $chart_object->detail_type;
        $chart_history->amount_type = DepositType::find($chart_object->id, 'credit');
        $chart_history->transaction_id = $bill->id;
        $chart_history->name = $bill->vendor_name;
        $chart_history->number = $bill->bill_number;
        $chart_history->amount = $amount;
        $chart_history->description = $request->notes;
        $chart_history->transaction_date = $this->formatDate($request->paid_at);
        $chart_history->save();

        $chartparents = app('App\Http\Controllers\Accounts\Accounting\Accounting')->parent_chartaccount_id($request->account_chart_id);
 
        CharthistoryParent::charthistoryparentsave($chart_history->id,$chartparents->original['parent_id_string'],$request->account_chart_id); 
        
        //History For Accounts Payable
        $chart =  Chartofaccounts::where('name','=','Accounts Payable')->first();
        $chart_receivable = [];
        $chart_receivable['company_id'] = session('company_id');
        $chart_receivable['transaction_id'] = $bill->id;
        $chart_receivable['transaction_type'] = 'Bill (Payment)';
        $chart_receivable['number'] = $bill->bill_number;
        $chart_receivable['currency_code'] = $currency->code;
        $chart_receivable['name'] =  $bill->vendor_name;
        $chart_receivable['amount_type'] = DepositType::find($chart->id, 'debit');
        $chart_receivable['chartofaccount_id'] = $chart->id;
        $chart_receivable['account_type_id'] = $chart->account_type_id;
        $chart_receivable['detail_type'] = $chart->detail_type;
        $chart_receivable['amount'] = $amount;
        $chart_receivable['description'] = $request->notes;
        $chart_receivable['transaction_date'] = $this->formatDate($request->paid_at);
        $chart_receivable_id = Charthistory::create($chart_receivable);

        $chartparents = app('App\Http\Controllers\Accounts\Accounting\Accounting')->parent_chartaccount_id($chart->id);
 
        CharthistoryParent::charthistoryparentsave($chart_receivable_id->id,$chartparents->original['parent_id_string'],$chart->id); 


        $bill_payment = BillPayment::create($request->input());

        // Upload attachment
        if ($request->file('attachment')) {
            $media = $this->getMedia($request->file('attachment'), 'bills');
            $bill_payment->attachMedia($media, 'attachment');
        }

        $request['status_code'] = $bill->bill_status_code;
        $request['notify'] = 0;

        $desc_amount = money((float) $request['amount'], (string) $request['currency_code'], true)->format();

        $request['description'] = $desc_amount . ' ' . trans_choice('general.payments', 1);

        BillHistory::create($request->input());

        $message = trans('messages.success.added', ['type' => trans_choice('general.payments', 1)]);

       if($request['bill_id_return']){
            \Session::flash('success', $message);
            DB::commit();
            return redirect('expenses/bills/makepayment/'.$request['bill_id']);
        }else{
            DB::commit();
            return response()->json([
                'success' => true,
                'error' => false,
                'message' => $message,
            ]);
       }        
    }

     /**
     * Bill Wise Payments Receive.
     *
     * @param  Bill Payment  $bill
     *
     * @return Response
    */
    public function paymentpage($id)
    {       
        $bill = Bill::with(['payments_total'])->find($id);

        $payment_methods = Modules::getPaymentMethods();

        $accounts = Account::whereEnabled(1)->get()->pluck('name_number', 'id');  
        
        return view('accounts.expenses.bills.payment',compact('bill','accounts','payment_methods'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  BillPayment  $payment
     *
     * @return Response
     */
    public function paymentDestroy(BillPayment $payment)
    {
        $bill = Bill::find($payment->bill_id);

        if ($bill->payments()->count() > 1) {
            $bill->bill_status_code = 'partial';
        } else {
            $bill->bill_status_code = 'received';
        }

        $bill->save();

        $desc_amount = money((float) $payment->amount, (string) $payment->currency_code, true)->format();

        $description = $desc_amount . ' ' . trans_choice('general.payments', 1);

        // Add bill history
        BillHistory::create([
            'company_id' => $bill->company_id,
            'bill_id' => $bill->id,
            'status_code' => $bill->bill_status_code,
            'notify' => 0,
            'description' => trans('messages.success.deleted', ['type' => $description]),
        ]);

        $payment->delete();

        $message = trans('messages.success.deleted', ['type' => trans_choice('general.bills', 1)]);

        \Session::flash('success',$message);

        return redirect()->back();
    }

    protected function prepareBill(Bill $bill)
    {
        $paid = 0;

        foreach ($bill->payments as $item) {
            $item->default_currency_code = $bill->currency_code;

            $paid += $item->getDynamicConvertedAmount();
        }

        $bill->paid = $paid;

        $bill->template_path = 'accounts.expenses.bills.bill';

        //event(new BillPrinting($bill));

        return $bill;
    }

    protected function addTotals($bill, $request, $sub_total)
    {
        $sort_order = 1;

        // Added bill total sub total
        BillTotal::create([
            'company_id' => $request['company_id'],
            'bill_id' => $bill->id,
            'code' => 'sub_total',
            'name' => 'bills.sub_total',
            'amount' => $sub_total,
            'sort_order' => $sort_order,
        ]);

        $sort_order++;

        // Added bill total total
        BillTotal::create([
            'company_id' => $request['company_id'],
            'bill_id' => $bill->id,
            'code' => 'total',
            'name' => 'bills.total',
            'amount' => $sub_total,
            'sort_order' => $sort_order,
        ]);
    }

    protected function getLogo($bill)
    {
        $logo = '';

        $media_id = setting('general.company_logo');

        if (isset($bill->vendor->logo) && !empty($bill->vendor->logo->id)) {
            $media_id = $bill->vendor->logo->id;
        }

        $media = Media::find($media_id);

        if (!empty($media)) {
            $path = storage_path($media->getDiskPath());

            if (!is_file($path)) {
                return $logo;
            }
        } else {
            //asset('public/accounts/img/company.png')
            $path = '';
            if (!is_file($path)) {
                return $logo;
            }
        }
        $image = Image::make($path)->encode()->getEncoded();

        if (empty($image)) {
            return $logo;
        }

        $extension = File::extension($path);

        $logo = 'data:image/' . $extension . ';base64,' . base64_encode($image);

        return $logo;
    }

    /**
     * Listing From Dashboard
     */
    public function dashboardList(GlobalRequest $request)
    {
        if ($request->type == 'all') {
            $data['title'] = ucwords('Total');
        } else {
            $data['title'] = ($request->type != 'over') ? ucwords($request->type) : ucwords($request->type . ' Due');
        }

        $data['type'] = $request->type;
        
        if ($request->daterange) {
            $selecteddate = explode('-', $request->daterange);
            $data['daterange'] = [date('Y-m-d', strtotime($selecteddate[0])), date('Y-m-d', strtotime($selecteddate[1]))];
        } else {
            $data['daterange'] = [];
        }

        if ($request->ajax()) {
            $bills = Bill::with(['vendor', 'status', 'items', 'payments', 'histories'])->orderBy('billed_at', 'desc');

            if (count($data['daterange']) != 0) {
                $range = $data['daterange'];
                $bills = $bills->whereBetween('billed_at', $range);
            }

            if ($data['type'] == 'opened') {
                $bills = $bills->where('bill_status_code', '!=', 'paid');
            }

            if ($data['type'] == 'over') {
                $bills = $bills->where('bill_status_code', '!=', 'paid')->whereDate('due_at', '<', date('Y-m-d'));
            }

            $bills = $bills->paginate(25);

            return view('accounts.dashboard.listings.expenses-ajax', compact('bills'));
        }

        return view('accounts.dashboard.listings.dashboard-expenses', $data);
    }
}
